<?php
/*
Plugin Name: Auto WebP Converter & Logger
Description: Converts uploaded images to WebP format and logs results.
Version: 1.4
Requires PHP: 7.0
Author: BabaPinnak
Author URI: https://babapinnak.com
License: GPL v2 or later
License URI: https://www.gnu.org/licenses/gpl-2.0.html
Text Domain: auto-webp-converter-logger
*/

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

class Autoweco_ConvertImagesToWebP {

    private $autoweco_options;
    private $autoweco_log_file;

    public function __construct() {
        $this->autoweco_options = get_option('autoweco_convert_images_to_webp_options');
        if (!is_array($this->autoweco_options)) {
            $this->autoweco_options = array(
                'quality' => 80,
                'delete_original' => false,
                'log_conversions' => false,
            );
            update_option('autoweco_convert_images_to_webp_options', $this->autoweco_options);
        }

        $upload_dir = wp_upload_dir();
        $log_dir = trailingslashit($upload_dir['basedir']) . 'auto-webp-converter-logger/';
        
        // Create the directory if it doesn't exist
        if (!file_exists($log_dir)) {
            wp_mkdir_p($log_dir);
        }
        
        $this->autoweco_log_file = $log_dir . 'webp-conversion-log.txt';
        
        

        add_action('admin_menu', array($this, 'autoweco_add_settings_page'));
        add_action('admin_init', array($this, 'autoweco_register_settings'));
        add_filter('wp_handle_upload', array($this, 'autoweco_convert_images_to_webp_on_upload'));
        add_filter('plugin_action_links_' . plugin_basename(__FILE__), array($this, 'autoweco_add_settings_link'));

		// 🛠️ Fix MIME type detection for .webp files
		add_action('add_attachment', array($this, 'autoweco_correct_attachment_metadata'), 10, 1);



		
        if (!function_exists('imagewebp')) {
            add_action('admin_notices', array($this, 'autoweco_gd_library_missing_notice'));
        }
    }


    public function autoweco_add_settings_link($links) {
        $settings_link = '<a href="' . esc_url(admin_url('options-general.php?page=autoweco_convert-images-to-webp')) . '">' . esc_html__('Settings', 'auto-webp-converter-logger') . '</a>';
        array_unshift($links, $settings_link);
        return $links;
    }

    public function autoweco_add_settings_page() {
        add_options_page(
            esc_html__('Auto WebP Converter Settings', 'auto-webp-converter-logger'),
            esc_html__('Images to WebP', 'auto-webp-converter-logger'),
            'manage_options',
            'autoweco_convert-images-to-webp',
            array($this, 'autoweco_create_settings_page')
        );
    }

    public function autoweco_create_settings_page() {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'auto-webp-converter-logger'));
        }
        ?>
        <div class="wrap">
            <h1><?php esc_html_e('Auto WebP Converter Settings', 'auto-webp-converter-logger'); ?></h1>
            <p><?php esc_html_e('Note: Only images smaller than 10MB will be automatically converted to WebP.', 'auto-webp-converter-logger'); ?></p>
            <form method="post" action="options.php">
                <?php
                settings_fields('autoweco_convert_images_to_webp_options_group');
                do_settings_sections('autoweco_convert-images-to-webp');
                submit_button();
                ?>
            </form>
        </div>
        <?php
    }

    public function autoweco_register_settings() {
        register_setting('autoweco_convert_images_to_webp_options_group', 'autoweco_convert_images_to_webp_options', 'autoweco_sanitize_options');

        add_settings_section('autoweco_settings_section', esc_html__('Settings', 'auto-webp-converter-logger'), null, 'autoweco_convert-images-to-webp');

        add_settings_field('quality', esc_html__('WebP Quality (0-100)', 'auto-webp-converter-logger'), array($this, 'autoweco_quality_callback'), 'autoweco_convert-images-to-webp', 'autoweco_settings_section');

        add_settings_field('delete_original', esc_html__('Delete Original Image', 'auto-webp-converter-logger'), array($this, 'autoweco_delete_original_callback'), 'autoweco_convert-images-to-webp', 'autoweco_settings_section');

        add_settings_field('log_conversions', esc_html__('Log Conversions to File', 'auto-webp-converter-logger'), array($this, 'autoweco_log_conversions_callback'), 'autoweco_convert-images-to-webp', 'autoweco_settings_section');
    }

    public function autoweco_quality_callback() {
        $options = get_option('autoweco_convert_images_to_webp_options');
        $quality = isset($options['quality']) ? (int) $options['quality'] : 80;
        printf('<input type="number" id="quality" name="autoweco_convert_images_to_webp_options[quality]" value="%s" min="0" max="100" class="small-text" />', esc_attr($quality));
    }

    public function autoweco_delete_original_callback() {
        $options = get_option('autoweco_convert_images_to_webp_options');
        $checked = !empty($options['delete_original']) ? 'checked' : '';
        printf('<input type="checkbox" id="delete_original" name="autoweco_convert_images_to_webp_options[delete_original]" %s />', esc_attr($checked));
    }

    public function autoweco_log_conversions_callback() {
        $options = get_option('autoweco_convert_images_to_webp_options');
        $checked = !empty($options['log_conversions']) ? 'checked' : '';
        printf('<input type="checkbox" id="log_conversions" name="autoweco_convert_images_to_webp_options[log_conversions]" %s />', esc_attr($checked));
    }

    private function autoweco_log_conversion($message) {
        if (empty($this->autoweco_options['log_conversions'])) {
            return;
        }

        $timestamp = current_time('mysql');
        $log_entry = sprintf("[%s] %s\n", $timestamp, $message);

        if (wp_is_writable(dirname($this->autoweco_log_file))) {
            file_put_contents($this->autoweco_log_file, $log_entry, FILE_APPEND | LOCK_EX);
        }
    }

    public function autoweco_gd_library_missing_notice() {
        if (current_user_can('manage_options')) {
            echo '<div class="notice notice-error"><p>' . esc_html__('Auto WebP Converter: GD library is not enabled. Image conversion will not work.', 'auto-webp-converter-logger') . '</p></div>';
        }
    }

    public function autoweco_convert_image_to_webp($image_path, $new_image_path, $image_ext) {
        if (!function_exists('imagewebp')) {
            $this->autoweco_log_conversion(__('Required GD library functions are missing.', 'auto-webp-converter-logger'));
            return false;
        }

        if (!is_readable($image_path)) {
            /* translators: %s: Path of the unreadable image */
            $this->autoweco_log_conversion(sprintf(__('Image not readable: %s.', 'auto-webp-converter-logger'), $image_path));
            return false;
        }

        switch (strtolower($image_ext)) {
            case 'jpeg':
            case 'jpg':
                $image = imagecreatefromjpeg($image_path);
                break;
            case 'png':
                $image = imagecreatefrompng($image_path);
                break;
            default:
                /* translators: %s: Unsupported image type */
                $this->autoweco_log_conversion(sprintf(__('Unsupported image type: %s.', 'auto-webp-converter-logger'), $image_path));
                return false;
        }

        if (!$image) {
            /* translators: %s: Path of the failed image creation */
            $this->autoweco_log_conversion(sprintf(__('Failed to create image from file: %s.', 'auto-webp-converter-logger'), $image_path));
            return false;
        }

        $result = imagewebp($image, $new_image_path, $this->autoweco_options['quality']);
        imagedestroy($image);

        if ($result) {
            /* translators: %s: Successfully converted image path */
            $this->autoweco_log_conversion(sprintf(__('Successfully converted: %s.', 'auto-webp-converter-logger'), $image_path));
            if (!empty($this->autoweco_options['delete_original'])) {
                wp_delete_file($image_path);
            }
        } else {
            /* translators: %s: Conversion failed image path */
            $this->autoweco_log_conversion(sprintf(__('Conversion failed for: %s.', 'auto-webp-converter-logger'), $image_path));
        }

        return $result;
    }

    public function autoweco_convert_images_to_webp_on_upload($file) {
        $max_file_size = 10 * 1024 * 1024; // 10MB

        if (!isset($file['file']) || !file_exists($file['file'])) {
            return $file;
        }

        if (filesize($file['file']) > $max_file_size) {
            /* translators: %s: File name too large */
            $this->autoweco_log_conversion(sprintf(__('File too large to convert: %s', 'auto-webp-converter-logger'), basename($file['file'])));
            return $file;
        }

        $filetype = wp_check_filetype($file['file']);

        if (empty($filetype['type']) || !in_array(strtolower($filetype['type']), ['image/jpg', 'image/jpeg', 'image/png'], true)) {
            return $file;
        }

        $upload_dir = wp_upload_dir();
        $webp_filename = preg_replace('/\.(jpg|jpeg|png)$/i', '.webp', basename($file['file']));
        $unique_webp_filename = wp_unique_filename($upload_dir['path'], $webp_filename);
        $webp_path = trailingslashit($upload_dir['path']) . $unique_webp_filename;

        if ($this->autoweco_convert_image_to_webp($file['file'], $webp_path, $filetype['ext'])) {
            $file['file'] = $webp_path;
            $file['url'] = trailingslashit($upload_dir['url']) . $unique_webp_filename;
            $file['type'] = 'image/webp';
        }

        return $file;
    }
	// 🛠️ Fix: Ensure .webp files are correctly detected with their MIME type
	public function autoweco_correct_attachment_metadata($post_ID) {
    $file = get_attached_file($post_ID);
    if (preg_match('/\.webp$/i', $file)) {
        // Update post_mime_type
        wp_update_post(array(
            'ID' => $post_ID,
            'post_mime_type' => 'image/webp',
        ));

        // Update attachment metadata
        update_post_meta($post_ID, '_wp_attached_file', _wp_relative_upload_path($file));
        }
    }
}

new Autoweco_ConvertImagesToWebP();

function autoweco_sanitize_options($input) {
    $new_input = array();
    $new_input['quality'] = (isset($input['quality']) && is_numeric($input['quality'])) ? min(max((int) $input['quality'], 0), 100) : 80;
    $new_input['delete_original'] = !empty($input['delete_original']);
    $new_input['log_conversions'] = !empty($input['log_conversions']);
    return $new_input;
}
?>
